************************************************************************
**                                                                    **
**      Submerged Aquatic Vegetation Subroutines for CE-QUAL-ICM      **
**                                                                    **
**         Multiple Depth Intervals from St Johns River Code          **
**       Multiple Species as per Original Chesapeake Bay Code         **
**       Transfers and Similar Properties Based on Julian Day         **
**                             May 9, 2006                            **
**                                                                    **
**                    Water Quality Modeling Group                    **
**                    U.S. Army Corps of Engineers                    **
**                    Waterways Experiment Station                    **
**                    Vicksburg, Mississippi 39180                    **
**                                                                    **
************************************************************************


************************************************************************
**             S U B R O U T I N E   S A V _ R E A D                  **
************************************************************************

      SUBROUTINE SAV_READ
      USE WQM; USE FILE_INFO
      IMPLICIT NONE
      CHARACTER*72 TITLE(6)
      CHARACTER*8  TIMVAR, PRINTIT, SPNIN
      REAL         KTPS1(NSAVP), KTPS2(NSAVP), KTBMSAV(NSAVP), 
     $             KTPE1, KTPE2, KTBME, KTPRE, TRPM(NSAVP),
     $             TRBMSAV(NSAVP)
      INTEGER      SP_IN

! vjp 8/23/05   added these declarations
      integer i, j, n, nsav
      real  trpe, trbme, trpre, tlook

C TITLE CARDS

      READ(SAVPF,1000)
      READ(SAVPF,1010) (TITLE(J),J=1,6)

C READ NUMBER OF DOMINANT SPECIES

      READ(SAVPF,1022) NDOMSP, FRDEPTH

C READ SPECIES NAME AND PROPERTIES

      DO 10010 I=1,NDOMSP
        READ(SAVPF,1044) SPNAM(I)
        READ(SAVPF,1030) PMSAV(I),PRSPSAVB(I),BMSAV(I),
     $    BMTBRREF(I),FDOSR(I),SALMAX(I),SAVDF(I)        
        READ(SAVPF,1030) KHNLEAF(I),KHNPSAV(I),KHNROOT(I),KHPLEAF(I),
     $    KHPROOT(I)
        READ(SAVPF,1030) ALPHSAV(I),KESAV(I),ALAC(I),ACAN(I),BCAN(I),
     $    DTHRESH(I),EFSAVSTL(I)
        READ(SAVPF,1030) TRPM(I),TRBMSAV(I)
        READ(SAVPF,1030) KTPS1(I),KTPS2(I),KTBMSAV(I)

C READ PIECEWISE FUNCTIONS DESCRIBING SLOUGHING, TRANSFER FROM SHOOTS
C TO ROOTS, TRANSFER FROM ROOTS TO SHOOTS

        READ(SAVPF,1040) TIMVAR, PRINTIT
        IF (TIMVAR .EQ. 'CONSTANT') THEN
          READ(SAVPF,1032) FPLEAF(1,I), FPSTEM(1,I),
     $      FPROOT(1,I), FPTUBER(1,I), TRTBRLF(1,I),SMLEAF(1,I)
            DO J=2,366
              FPLEAF(J,I) = FPLEAF(1,I)
              FPSTEM(J,I) = FPSTEM(1,I)
              FPROOT(J,I) = FPROOT(1,I)
              FPTUBER(J,I)= FPTUBER(1,I)
              TRTBRLF(J,I)= TRTBRLF(1,I)
              SMLEAF(J,I) = SMLEAF(1,I)
            END DO
        ELSE
          READ(SAVPF,1032) (FPLEAF(J,I), FPSTEM(J,I),
     $      FPROOT(J,I), FPTUBER(J,I), TRTBRLF(J,I),SMLEAF(J,I),
     $      J=1,366)
        END IF
          
10010 CONTINUE

        READ(SAVPF,1030) ACDWSAV,ANDWSAV,APDWSAV

C COMPUTE ADDITIONAL STOICHIOMETRY OF SAV

        ANCSAV = ANDWSAV/ACDWSAV
        APCSAV = APDWSAV/ACDWSAV

C READ DISTRIBUTION OF SAV BIOMASS AFTER MORTALITY

        READ(SAVPF,1030) FNISAV,FNLDSAV,FNRDSAV,FNLPSAV,FNRPSAV,FNG3SAV
        READ(SAVPF,1030) FRNSAV(1),FRNSAV(2),FRNSAV(3)
        READ(SAVPF,1030) FPISAV,FPLDSAV,FPRDSAV,FPLPSAV,FPRPSAV,FPG3SAV
        READ(SAVPF,1030) FRPSAV(1),FRPSAV(2),FRPSAV(3)
        READ(SAVPF,1030) FDOSAV,FCLDSAV,FCRDSAV,FCLPSAV,FCRPSAV,FCG3SAV
        READ(SAVPF,1030) FRCSAV(1),FRCSAV(2),FRCSAV(3)

C READ EFFECT OF SAV ON NET SETTLING TO SEDIMENTS

        READ(SAVPF,1030) WSSSAV,WSLSAV,WSRSAV,WSG3SAV,WS1SAV,WS2SAV,
     $    WS3SAV,WSPIPSAV

C READ EPIPHYTE NAME AND PROPERTIES

      READ(SAVPF,1040) EPINAM
      READ(SAVPF,1030) PMEPI,PRSPEPI,BMEPI,PREPI
      READ(SAVPF,1030) CCHLEPI,ADWCEPI,ANCEPI,APCEPI
      READ(SAVPF,1030) KHNEPI,KHNPEPI,KHPEPI
      READ(SAVPF,1030) ALPHEPI,KEEPI
      READ(SAVPF,1030) TRPE,TRBME,TRPRE
      READ(SAVPF,1030) KTPE1,KTPE2,KTBME,KTPRE
      READ(SAVPF,1030) KHEP

C READ DISTRIBUTION OF EPIPHYTE BIOMASS AFTER MORTALITY

      READ(SAVPF,1030) FNIEPI,FNLDEPI,FNRDEPI,FNLPEPI,FNRPEPI,FNG3EPI
      READ(SAVPF,1030) FNIPEP,FNLDPEP,FNRDPEP,FNLPPEP,FNRPPEP,FNG3PEP
      READ(SAVPF,1030) FPIEPI,FPLDEPI,FPRDEPI,FPLPEPI,FPRPEPI,FPG3EPI
      READ(SAVPF,1030) FPIPEP,FPLDPEP,FPRDPEP,FPLPPEP,FPRPPEP,FPG3PEP
      READ(SAVPF,1030) FCLDEPI,FCRDEPI
      READ(SAVPF,1030) FCLDPEP,FCRDPEP,FCLPPEP,FCRPPEP,FCG3PEP
      
C ZERO OUT SAV AREAS ETC.

      DO B=1, NSB
         SAVCELL(B) = 0
         SAVDPH(B) = 0.0
	 SAVFRAC(B)= 0.0
	 SAVAREA(B)  = 0.0
	 SAVEFCT(B)  = 1.0
      ENDDO

C READ NUMBER OF CELLS WITH SAV

      READ(SAVPF,1020) NSAVCELL

C READ SAV CELLS, AREAS, AND DEPTHS
      
      READ(SAVPF,1006)
      DO I=1,NSAVCELL
        READ(SAVPF,*) B, DOMSP(B), SAVAREA_NX(B), SAVDPH_NX(B)
        SAVCELL(I) = B
      END DO

***** Input FORMAT statements

 1000 FORMAT(///)
 1006 FORMAT(/)
 1010 FORMAT(A72)
 1020 FORMAT(//(8X,9I8))
 1022 FORMAT(//(8X,I8,A8,I8,I8))
 1030 FORMAT(//(8X,9F8.0))
 1032 FORMAT(//(8X,6F8.0))
 1040 FORMAT(//8X,2A8)
 1042 FORMAT(//8X,I8,A8)
 1044 FORMAT(//8X,A8,I8)
 1050 FORMAT(//(8X,A8,F8.0))

C OUTPUT WHAT WAS INPUT

      OPEN (SVO,FILE=SVOFN)
      WRITE(SVO,2010) (TITLE(J),J=1,6)

C WRITE NUMBER OF DOMINANT SPECIES

      WRITE(SVO,2022) FRDEPTH
      WRITE(SVO,2020) NDOMSP

C WRITE SAV SPECIES NAME AND PROPERTIES

      DO 20010 I=1,NDOMSP
        WRITE(SVO,2040) SPNAM(I), I
        WRITE(SVO,2050) PMSAV(I),TRPM(I),KTPS1(I),KTPS2(I)
        WRITE(SVO,2052) PMSAV(I)/ACDWSAV,TRPM(I)
        WRITE(SVO,4010) KHNLEAF(I),KHNPSAV(I),KHNROOT(I),KHPLEAF(I),
     $    KHPROOT(I)
        WRITE(SVO,2060) BMSAV(I),TRBMSAV(I),KTBMSAV(I)
        WRITE(SVO,2058) PRSPSAVB(I),SALMAX(I)
        WRITE(SVO,3070) FDOSR(I)
        WRITE(SVO,3000) ALPHSAV(I)
        WRITE(SVO,3010) KESAV(I),ALAC(I),ACAN(I),BCAN(I)
	IF (FRDEPTH .NE. ' FRDEPTH') WRITE(SVO,3017) SAVDF(I)
	WRITE(SVO,3016) DTHRESH(I),EFSAVSTL(I)
        IF (PRINTIT .EQ. '     ALL') THEN
          WRITE(SVO,3080)
          WRITE(SVO,3090) (J,FPLEAF(J,I),FPSTEM(J,I),FPROOT(J,I),
     $      FPTUBER(J,I),TRTBRLF(J,I),SMLEAF(J,I),J=1,366)
        END IF
20010 CONTINUE

      WRITE(SVO,2056) ACDWSAV,ANDWSAV,APDWSAV

C WRITE DISTRIBUTION OF BIOMASS UPON MORTALITY

      WRITE(SVO,4020)
      WRITE(SVO,4030) FNISAV,FNLDSAV,FNRDSAV,FNLPSAV,FNRPSAV,FNG3SAV
      WRITE(SVO,4040) FPISAV,FPLDSAV,FPRDSAV,FPLPSAV,FPRPSAV,FPG3SAV
      WRITE(SVO,4050) FDOSAV,FCLDSAV,FCRDSAV,FCLPSAV,FCRPSAV,FCG3SAV
      WRITE(SVO,4080)
      WRITE(SVO,4090) FRNSAV(1),FRNSAV(2),FRNSAV(3)
      WRITE(SVO,5000) FRPSAV(1),FRPSAV(2),FRPSAV(3)
      WRITE(SVO,5010) FRCSAV(1),FRCSAV(2),FRCSAV(3)

C WRITE EFFECT OF SAV ON NET SETTLING TO SEDIMENTS
      WRITE(SVO,5060) WSSSAV,WSLSAV,WSRSAV,WSG3SAV,WS1SAV,WS2SAV,
     $                WS3SAV

C WRITE EPIPHYTE PROPERTIES

      WRITE(SVO,2040) EPINAM
      WRITE(SVO,2051) PMEPI,TRPE,KTPE1,KTPE2
      WRITE(SVO,2052) PMEPI/CCHLEPI,TRPE
      WRITE(SVO,2062) CCHLEPI,ADWCEPI,ANCEPI,APCEPI
      WRITE(SVO,5020) KHNEPI,KHNPEPI,KHPEPI
      WRITE(SVO,2065) BMEPI,TRBME,KTBME
      WRITE(SVO,3020) PREPI,TRPRE,KTPRE
      WRITE(SVO,3002) ALPHEPI
      WRITE(SVO,3030) KEEPI
      WRITE(SVO,4000) KHEP

C WRITE DISTRIBUTION OF BIOMASS UPON MORTALITY

      WRITE(SVO,5030)
      WRITE(SVO,4030) FNIEPI,FNLDEPI,FNRDEPI,FNLPEPI,FNRPEPI,FNG3EPI
      WRITE(SVO,4040) FPIEPI,FPLDEPI,FPRDEPI,FPLPEPI,FPRPEPI,FPG3EPI
      WRITE(SVO,5050) FCLDEPI,FCRDEPI
      WRITE(SVO,5040)
      WRITE(SVO,4030) FNIPEP,FNLDPEP,FNRDPEP,FNLPPEP,FNRPPEP,FNG3PEP
      WRITE(SVO,4040) FPIPEP,FPLDPEP,FPRDPEP,FPLPPEP,FPRPPEP,FPG3PEP
      WRITE(SVO,4070) FCLDPEP,FCRDPEP,FCLPPEP,FCRPPEP,FCG3PEP
      
C WRITE NUMBER OF SAV CELLS, SAV WIDTH AND DEPTH INCREMENTS

cvjp modified 11/3/2005
      WRITE(SVO,5070) NSAVCELL
      IF (PRINTIT .EQ. '     ALL') THEN
        DO I=1,NSAVCELL
          B = SAVCELL(I)
          WRITE(SVO,5080) I, B, DOMSP(B), SAVAREA_NX(B), SAVDPH_NX(B) 
        END DO
      END IF
      
C CREATE LOOKUP TABLE OF TEMPERATURE EFFECTS

      DO I = 1,NDOMSP
        DO J = -50,400
          TLOOK = FLOAT(J)/10.

          IF (TLOOK.LT.TRPM(I)) THEN
            FTPSAV(J,I) = EXP(-KTPS1(I)*(TLOOK-TRPM(I))**2)
          ELSE
            FTPSAV(J,I) = EXP(-KTPS2(I)*(TLOOK-TRPM(I))**2)
          END IF

          FTRSAV(J,I) = EXP(KTBMSAV(I)*(TLOOK-TRBMSAV(I)))

        END DO
      END DO

      DO J = -50,400
        TLOOK = FLOAT(J)/10.

        IF (TLOOK.LT.TRPE) THEN
          FTPEP(J) = EXP(-KTPE1*(TLOOK-TRPE)**2)
        ELSE
          FTPEP(J) = EXP(-KTPE2*(TLOOK-TRPE)**2)
        END IF
        
        FTREP(J)  = EXP(KTBME*(TLOOK-TRBME))
        FTPREP(J) = EXP(KTPRE*(TLOOK-TRPRE))

      END DO
      
C ZERO OUT ALL COUNTERS FOR EFECTS ON WATER COLUMN AND SEDIMENTS

      DO B=1,NBB

        DOSAV(B)   = 0.
        LDOCSAV(B) = 0.
        RDOCSAV(B) = 0.
        LPOCSAV(B) = 0.
        RPOCSAV(B) = 0.
        G3CSAV(B)  = 0.

        DOEPI(B)   = 0.
        LDOCEPI(B) = 0.
        RDOCEPI(B) = 0.
        LPOCEPI(B) = 0.
        RPOCEPI(B) = 0.
        G3CEPI(B)  = 0.

        NH4SAVW(B)  = 0.
        NO3SAVW(B)  = 0.
        LDONSAVW(B) = 0.
        RDONSAVW(B) = 0.
        LPONSAVW(B) = 0.
        RPONSAVW(B) = 0.
        G3NSAVW(B)  = 0.

        NH4EPI(B)  = 0.
        NO3EPI(B)  = 0.
        RDONEPI(B) = 0.
        LDONEPI(B) = 0.
        LPONEPI(B) = 0.
        RPONEPI(B) = 0.
        G3NEPI(B)  = 0.

        PO4SAVW(B)  = 0.
        LDOPSAVW(B) = 0.
        RDOPSAVW(B) = 0.
        LPOPSAVW(B) = 0.
        RPOPSAVW(B) = 0.
        G3PSAVW(B)  = 0.
        
        PO4EPI(B)  = 0.
        LDOPEPI(B) = 0.
        RDOPEPI(B) = 0.
        LPOPEPI(B) = 0.
        RPOPEPI(B) = 0.
        G3PEPI(B)  = 0.
        
      END DO

      DO B=1,NBB

        SEDDOSAV(B) = 0.
        SEDCSAV(B)  = 0. 
        SEDNH4SAV(B) = 0.
        SEDPO4SAV(B) = 0.
        SEDNSAV(B)   = 0.
        SEDPSAV(B)   = 0.

      END DO
      
***** Output FORMAT statements

 2010 FORMAT(1X,A72)
 2020 FORMAT(/' THERE ARE ',I3,' DOMINANT SPECIES WITH PROPERTIES:') 
 2022 FORMAT(/' SAV AFFECTS TAU BASED ON ',A8) 
 2040 FORMAT(/1X,A8,I8)
 2050 FORMAT(' PRODUCTION = ',F8.3,' GM C/GM DW/DAY AT ',F8.2,' C.'/
     $  ' KT1 = ',F8.3,' KT2 = ',F8.3,' PER DEGREE**2')
 2051 FORMAT(' PRODUCTION = ',F8.3,' GM C/GM CHL/DAY AT ',F8.2,' C.'/
     $  ' KT1 = ',F8.3,' KT2 = ',F8.3,' PER DEGREE**2')
 2052 FORMAT(' CARBON SPECIFIC GROWTH RATE = ',F8.3,' PER DAY AT ',
     $  F8.2,' C.')
 2056 FORMAT(' CARBON TO DRY WEIGHT RATIO = ',F8.3/
     $  ' NITROGEN TO DRY WEIGHT RATIO = ',F8.3/
     $  ' PHOSPHORUS TO DRY WEIGHT RATIO = ',F8.3)
 2058 FORMAT(' PHOTORESPIRATION = ',F8.3,' * PRODUCTION @ ZERO SALT'/
     $  ' PHOTORESPIRATION = 1.0 WHEN SALINITY > ',F8.3)     
 2060 FORMAT(/' PLANT RESPIRATION = ',F8.3,' PER DAY AT ',F8.2,
     $  ' C.  KT = ',F8.3,' PER DEGREE')
 2062 FORMAT(' CARBON TO CHLOROPHYLL RATIO = ',F8.3/
     $  ' DRY WEIGHT TO CARBON RATIO = ',F8.3/
     $  ' NITROGEN TO CARBON RATIO = ',F8.3/
     $  ' PHOSPHORUS TO CARBON RATIO = ',F8.3)
 2065 FORMAT(/' RESPIRATION = ',F8.3,' PER DAY AT ',F8.2,
     $  ' C.  KT = ',F8.3,' PER DEGREE')
 3000 FORMAT(/' ALPHA = ',F8.3,' (GM C/GM DW) / (E/M**2)')
 3002 FORMAT(/' ALPHA = ',F8.3,' (GM C/GM CHL) / (E/M**2)')
 3010 FORMAT(' LIGHT ATTENUATION = ',F8.3,' M**2/GM SHOOT C',/
     $  ' M**2 LEAF AREA PER GM LEAF CARBON  = ',F8.3,/
     $  ' CANOPY HEIGHT = ',F8.3,' + ',F8.3,' * SHOOTS (M)')
 3012 FORMAT(' GROWTH PROBABILITY NOT ENABLED ')
 3014 FORMAT(' GROWTH PROBABILITY = 1.0'
     $ ' WHEN THRESHOLD EXCEEDS ',F6.1,' G ROOTS / SQ M',/
     $ I5,' PROBABILITIES INPUT')
 3016 FORMAT(' WHEN DENSITY > ',F6.1,' G C / SQ M,'
     $  ' SETTLING IS MULTIPLIED BY ',F6.2)
 3017 FORMAT(' SAV EFFECT ON BOTTOM SEHEAR STRESS (SQ M/G C)',
     $ F8.3)
 3020 FORMAT(' PREDATION = ',F8.3,' PER DAY AT ',F8.2,
     $  ' C.  KT = ',F8.3,' PER DEGREE')
 3030 FORMAT(' LIGHT ATTENUATION = ',F8.3,
     $  ' M**2 LEAF AREA PER GM EPIPHYTE C'/)
 3040 FORMAT(' SPECIES SPECIFIED INCORRECTLY IN CELL ',I5,
     $  ' AS ',A8)
 3070 FORMAT(' FRACTION DO TRANSFERRED FROM SHOOTS TO ROOTS =',F8.3)
 3080 FORMAT(/'     DAY      FPLEAF    FPSTEM    FPROOT   FPTUBER',
     $'   TRTBRLF','    SMLEAF')
 3090 FORMAT(I7,3X,6F10.3)
 4000 FORMAT(/' EPIPHYTE DENSITY AT WHICH GROWTH IS HALVED = ',F8.3,
     $  ' GM C/GM C')
 4010 FORMAT(/' NITROGEN HALF-SATURATION CONC (SHOOTS) = ',F8.3,
     $  ' GM/M**3'/
     $  ' NH4 HALF-SATURATION CONC (PREFERENCE) = ',F8.3,' GM/M**3'/
     $  ' NITROGEN HALF-SATURATION CONC (ROOTS) = ',F8.3,' GM/M**3'/
     $  ' PHOSPHORUS HALF-SATURATION CONC (SHOOTS) = ',F8.3,
     $  ' GM/M**3'/
     $  ' PHOSPHORUS HALF-SATURATION CONC (ROOTS) = ',F8.3,
     $  ' GM/M**3')
 4020 FORMAT(/' DISTRIBUTION OF SAV UPON MORTALITY'/
     $  ' SHOOT RESPIRATION  DIS INORG  LAB DISS  REF DISS  ',
     $  'LAB PART  REF PART   G3 PART')
 4030 FORMAT(' NITROGEN          ',6F10.3)
 4040 FORMAT(' PHOSPHORUS        ',6F10.3)
 4050 FORMAT(' CARBON            ',6F10.3)
 4070 FORMAT(' CARBON            ',10X,5F10.3)
 4080 FORMAT(/' ROOT MORTALITY   LAB PART  REF PART     INERT')
 4090 FORMAT(' NITROGEN       ',3F10.3)
 5000 FORMAT(' PHOSPHORUS     ',3F10.3)
 5010 FORMAT(' CARBON         ',3F10.3)
 5020 FORMAT(/' NITROGEN HALF-SATURATION CONC = ',F8.3,' GM/M**3'/
     $  ' NH4 HALF-SATURATION CONC (PREFERENCE) = ',F8.3,' GM/M**3'/
     $  ' PHOSPHORUS HALF-SATURATION CONC = ',F8.3,' GM/M**3')
 5030 FORMAT(/' DISTRIBUTION OF EPIPHYTES UPON MORTALITY'/
     $  ' RESPIRATION        DIS INORG  LAB DISS  REF DISS  ',
     $  'LAB PART  REF PART   G3 PART')
 5040 FORMAT(' PREDATION')
 5050 FORMAT(' CARBON            ',10X,2F10.3)
 5060 FORMAT(/'EFFECT OF SAV ON NET SETTLING TO SEDIMENTS ',
     $'(M**3/GM C/DAY)'/'  SOLIDS  LABILE REFRACT G3 PART      B1',
     $'      B2      B3'/7F8.2)
 5070 FORMAT(/' NUMBER OF SAV CELLS = ',I8/)
 5080 FORMAT(' SAV CELL ',I6,' WQM CELL ',I6, ' DOM SP ',I4,
     $'  AREA ',F8.0,' DEPTH ',F5.2)
 
 
      RETURN
      END

************************************************************************
**                    S U B R O U T I N E   S A V                     **
************************************************************************

      SUBROUTINE SAV_COMP
      USE WQM
      IMPLICIT NONE
      INTEGER JD
      REAL DENLIM(NSBP), PRSPSAV(NSBP)
      REAL IK, IATCNPY, IAVSH, IAVEP, NSED, NWAT, LFOLD
      REAL NPRSAV(NSBP), NPREPI(NSBP)
      REAL HFRAC(NSBP), AVGDEN(NSBP)
cvjp 9/20/05 added declarations
      integer itemp, j, n, i
      real atn, psed, df, pwat, hsmol, stest, glimsav, glimepi, dellf
      real dltdy, tdiff, zmean, ztocnpy

C CONVERT MODEL TIME STEP TO DAYS, DETERMINE JULIAN DAY

      DLTDY = DLT/86400.
      JD    = 1.0 + AMOD(JDAY,365.25)
      
C ZERO OUT EFFECTS OF SAV CELLS ON WATER COLUMN AND SEDIMENTS

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)      
        DOSAV(B)   = 0.
        LDOCSAV(B) = 0.
        RDOCSAV(B) = 0.
        LPOCSAV(B) = 0.
        RPOCSAV(B) = 0.
        G3CSAV(B)  = 0.

        DOEPI(B)   = 0.
        LDOCEPI(B) = 0.
        RDOCEPI(B) = 0.
        LPOCEPI(B) = 0.
        RPOCEPI(B) = 0.
        G3CEPI(B)  = 0.

        NH4SAVW(B)  = 0.
        NO3SAVW(B)  = 0.
        LDONSAVW(B) = 0.
        RDONSAVW(B) = 0.
        LPONSAVW(B) = 0.
        RPONSAVW(B) = 0.
        G3NSAVW(B)  = 0.

        NH4EPI(B)  = 0.
        NO3EPI(B)  = 0.
        RDONEPI(B) = 0.
        LDONEPI(B) = 0.
        LPONEPI(B) = 0.
        RPONEPI(B) = 0.
        G3NEPI(B)  = 0.

        PO4SAVW(B)  = 0.
        LDOPSAVW(B) = 0.
        RDOPSAVW(B) = 0.
        LPOPSAVW(B) = 0.
        RPOPSAVW(B) = 0.
        G3PSAVW(B)  = 0.
        
        PO4EPI(B)  = 0.
        LDOPEPI(B) = 0.
        RDOPEPI(B) = 0.
        LPOPEPI(B) = 0.
        RPOPEPI(B) = 0.
        G3PEPI(B)  = 0.
        
      END DO

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)      
        SEDDOSAV(B)  = 0.
        SEDCSAV(B)   = 0. 
        SEDNH4SAV(B) = 0.
        SEDPO4SAV(B) = 0.
        SEDNSAV(B)   = 0.
        SEDPSAV(B)   = 0.
      END DO

      DO I=1,NSAVCELL
        B = SAVCELL(I)      
      END DO

C LIGHT EFFECTS

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)
	J = DOMSP(B)
        ITEMP = 10. * T(B) + 0.05
        
C COMPUTE IRRADIANCE AT CANOPY HEIGHT

          ZMEAN   = SAVDPH(B)  
          HCAN = ACAN(J) + BCAN(J)*(LEAF(B)+STEM(B)) 
          HCAN = MIN(ZMEAN,HCAN)
          ZTOCNPY = ZMEAN-HCAN
          WATATN(B) = EXP(-KESS(B)*ZTOCNPY)
          IATCNPY = I0*WATATN(B)

C COMPUTE ATTENUATION BY EPIPHYTES AND SELF SHADING

          EPATN(B) = EXP(-KEEPI*ADWCEPI*EP(B))
          ATN = KESS(B)*HCAN+KESAV(J)*(LEAF(B)+STEM(B))
          SAVATN(B) = (1.-EXP(-ATN))/(ATN+1.0E-6)
          IAVEP = IATCNPY*SAVATN(B)
          IAVSH = IAVEP*EPATN(B)

C COMPUTE LIGHT LIMITATIONS TO GROWTH

          IK = PMSAV(J)*FTPSAV(ITEMP,J)/ALPHSAV(J)
          FISH(B) = IAVSH/(SQRT(IK*IK+IAVSH*IAVSH)+1.0E-30)
          IK = PMEPI*FTPEP(ITEMP)/ALPHEPI
          FIEP(B) = IAVEP/(SQRT(IK*IK+IAVEP*IAVEP)+1.0E-30)
          
      END DO    ! end light affects loop

C COMPUTE NUTRIENT LIMITATIONS TO GROWTH, SAV FIRST

cvjp modified 11/3/2005
      DO I=1,NSAVCELL

        B = SAVCELL(I)
	J = DOMSP(B)
        NSED = MAX(0.,NH4T2TM1S(B)/1000.)
        NWAT = MAX(0.,NH4(B)+NO3(B))
        PSED = MAX(0.,PO4T2TM1S(B)/1000.)
        DF = 1./(1.+KADPO4*SSI(B))
        PWAT = MAX(0.,DF*PO4(B))
        
        
        NLSAV(B) = (KHNROOT(J)*NWAT+KHNLEAF(J)*NSED)/
     $    (KHNLEAF(J)*KHNROOT(J)+KHNROOT(J)*NWAT+KHNLEAF(J)*NSED)
        FNSEDSAV(B) = (KHNLEAF(J)*NSED)/
     $    (KHNROOT(J)*NWAT+KHNLEAF(J)*NSED+1.0E-6)
        NPRSAV(B) = NH4(B)/(KHNPSAV(J)+NO3(B)+1.E-6)*
     $   (NO3(B)/(KHNPSAV(J)+NH4(B)+1.E-6)
     $   +KHNPSAV(J)/(1.E-6+NH4(B)+NO3(B)))

        PLSAV(B) = (KHPROOT(J)*PWAT+KHPLEAF(J)*PSED)/
     $    (KHPLEAF(J)*KHPROOT(J)+KHPROOT(J)*PWAT+KHPLEAF(J)*PSED)
        FPSEDSAV(B) = (KHPLEAF(J)*PSED)/
     $    (KHPROOT(J)*PWAT+KHPLEAF(J)*PSED+1.0E-6)

C NOW EPIPHYTES

        NLEPI(B) = NWAT/(KHNEPI+NWAT+1.0E-6)
        NPREPI(B) = NH4(B)/(KHNPEPI+NO3(B)+1.E-6)*
     $   (NO3(B)/(KHNPEPI+NH4(B)+1.E-6)
     $    +KHNPEPI/(1.E-6+NH4(B)+NO3(B)))
        PLEPI(B) = PWAT/(KHPEPI+PWAT+1.0E-6)
      
C COMPUTE EPIPHYTE DENSITY LIMITATION

        DENLIM(B) = KHEP/(KHEP+EP(B)+1.0E-6)

      END DO     ! end nutrient limitations loop

cvjp modified 11/3/2005
      DO I=1,NSAVCELL

        B = SAVCELL(I)
	J = DOMSP(B)
        ITEMP = 10. * T(B) + 0.05
        
C COMPUTE RESPIRATION.

        STEST    = MAX(SALT(B),0.0)
        PRSPSAV(B) = PRSPSAVB(J) + (1.-PRSPSAVB(J)) * 
     $    0.5 * (1.+TANH(STEST-SALMAX(J)))
        BMLEAF(B)= BMSAV(J)*FTRSAV(ITEMP,J)+SMLEAF(JD,J)
        BMSTEM(B)= BMSAV(J)*FTRSAV(ITEMP,J)+SMLEAF(JD,J)
        SLSH(B)  = BMSAV(J)*FTRSAV(ITEMP,J)*(FCLPSAV+FCRPSAV+FCG3SAV)
     $      +SMLEAF(JD,J)    
        BMROOT(B)= BMSAV(J)*FTRSAV(ITEMP,J)
        BMTUBER(B)= BMTBRREF(J)*FTRSAV(ITEMP,J)

        GLIMSAV = MIN(NLSAV(B),PLSAV(B),FISH(B))
        GLIMEPI = MIN(NLEPI(B),PLEPI(B),FIEP(B))
        PLEAF(B) = PMSAV(J)*FTPSAV(ITEMP,J)*GLIMSAV/ACDWSAV

        NPPSAV(B) = (PLEAF(B)-BMLEAF(B))*LEAF(B)

C EPIPHYTE GROWTH AND RESPIRATION
C PHOTORESPIRATION IS NOT CONSISTENT WITH SAV
C FOR NOW LEAVE IT ALONE

        PEP(B) = PMEPI*FTPEP(ITEMP)*GLIMEPI*DENLIM(B)/CCHLEPI
        BMEP(B) = BMEPI*FTREP(ITEMP)+PRSPEPI*PEP(B)
        PREP(B) = PREPI*EP(B)*FTPREP(ITEMP)
        NPPEPI(B) = (PEP(B)-BMEP(B))*EP(B)

      END DO

C INTERACTIONS WITH WATER COLUMN

cvjp modified 11/3/2005
      DO I=1,NSAVCELL

        B = SAVCELL(I)
	J = DOMSP(B)

C DISSOLVED OXYGEN AND CARBON

          DOSAV(B) = DOSAV(B)+SAVFRAC(B)*
     $      (AOCR*(1.-FDOSR(J))*PLEAF(B)*LEAF(B)
     $      -AOCR*((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))*FDOSAV)
          LDOCSAV(B) = LDOCSAV(B)+SAVFRAC(B)*
     $      ((LEAF(B)*(BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      +STEM(B)*BMSTEM(B))*FCLDSAV)
          RDOCSAV(B) = RDOCSAV(B)+SAVFRAC(B)*
     $      ((LEAF(B)*(BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      +STEM(B)*BMSTEM(B))*FCRDSAV)
          LPOCSAV(B) = LPOCSAV(B)+SAVFRAC(B)*
     $      ((LEAF(B)*(BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      +STEM(B)*BMSTEM(B))*FCLPSAV)
          RPOCSAV(B) = RPOCSAV(B)+SAVFRAC(B)*
     $      ((LEAF(B)*(BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      +STEM(B)*BMSTEM(B))*FCRPSAV)
          G3CSAV(B)  =  G3CSAV(B)+SAVFRAC(B)*
     $      ((LEAF(B)*(BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      +STEM(B)*BMSTEM(B))*FCG3SAV)


          DOEPI(B) = DOEPI(B)+SAVFRAC(B)*EP(B)*LEAF(B)*ALAC(J)
     $      *AOCR*(PEP(B)-BMEP(B)*(1.-FCLDEPI-FCRDEPI))
          LDOCEPI(B) = LDOCEPI(B)+SAVFRAC(B)*
     $      (EP(B)*LEAF(B)*ALAC(J)*BMEP(B)*FCLDEPI
     $      + EP(B)*LEAF(B)*ALAC(J)*(PREP(B)+SLSH(B))*FCLDPEP)
          RDOCEPI(B) = RDOCEPI(B)+SAVFRAC(B)*
     $      (EP(B)*LEAF(B)*ALAC(J)*BMEP(B)*FCRDEPI
     $      + EP(B)*LEAF(B)*ALAC(J)*(PREP(B)+SLSH(B))*FCRDPEP)
          LPOCEPI(B) = LPOCEPI(B)+SAVFRAC(B)*
     $      (EP(B)*LEAF(B)*ALAC(J)*(PREP(B)+SLSH(B))*FCLPPEP)
          RPOCEPI(B) = RPOCEPI(B)+SAVFRAC(B)*
     $      (EP(B)*LEAF(B)*ALAC(J)*(PREP(B)+SLSH(B))*FCRPPEP)
          G3CEPI(B) = G3CEPI(B)+SAVFRAC(B)*
     $      (EP(B)*LEAF(B)*ALAC(J)*(PREP(B)+SLSH(B))*FCG3PEP)


C NITROGEN AND PHOSPHORUS

          NH4SAVW(B) = NH4SAVW(B)+SAVFRAC(B)*
     $      (ANCSAV*FNISAV*((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))-
     $      (1.-FNSEDSAV(B))*NPRSAV(B)*PLEAF(B)*ANCSAV*LEAF(B))
          NO3SAVW(B) = NO3SAVW(B)-SAVFRAC(B)*
     $      (1.-FNSEDSAV(B))*(1.-NPRSAV(B))*PLEAF(B)*
     $      ANCSAV*LEAF(B)
          LDONSAVW(B) = LDONSAVW(B)+SAVFRAC(B)*FNLDSAV*ANCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          RDONSAVW(B) = RDONSAVW(B)+SAVFRAC(B)*FNRDSAV*ANCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          LPONSAVW(B) = LPONSAVW(B)+SAVFRAC(B)*FNLPSAV*ANCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          RPONSAVW(B) = RPONSAVW(B)+SAVFRAC(B)*FNRPSAV*ANCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          G3NSAVW(B)  =  G3NSAVW(B)+SAVFRAC(B)*FNG3SAV*ANCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))


          NH4EPI(B) = NH4EPI(B)+SAVFRAC(B)*
     $      (FNIEPI*BMEP(B)+FNIPEP*(PREP(B)+SLSH(B))
     $      -NPREPI(B)*PEP(B))*ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          NO3EPI(B) = NO3EPI(B)-SAVFRAC(B)*(1.-NPREPI(B))*PEP(B)
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          RDONEPI(B) = RDONEPI(B)+SAVFRAC(B)*
     $      (FNRDEPI*BMEP(B)+FNRDPEP*(PREP(B)+SLSH(B)))
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          LDONEPI(B) = LDONEPI(B)+SAVFRAC(B)*
     $      (FNLDEPI*BMEP(B)+FNLDPEP*(PREP(B)+SLSH(B)))
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          LPONEPI(B) = LPONEPI(B)+SAVFRAC(B)*
     $      (FNLPEPI*BMEP(B)+FNLPPEP*(PREP(B)+SLSH(B)))
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          RPONEPI(B) = RPONEPI(B)+SAVFRAC(B)*
     $      (FNRPEPI*BMEP(B)+FNRPPEP*(PREP(B)+SLSH(B)))
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)
          G3NEPI(B)  =  G3NEPI(B)+SAVFRAC(B)*
     $      (FNG3EPI*BMEP(B)+FNG3PEP*(PREP(B)+SLSH(B)))
     $      *ANCEPI*EP(B)*LEAF(B)*ALAC(J)


          PO4SAVW(B) = PO4SAVW(B)+SAVFRAC(B)*(APCSAV*FPISAV*
     $      ((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
     $      -(1.-FPSEDSAV(B))*PLEAF(B)*APCSAV*LEAF(B))
          LDOPSAVW(B) = LDOPSAVW(B)+SAVFRAC(B)*FPLDSAV*APCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          RDOPSAVW(B) = RDOPSAVW(B)+SAVFRAC(B)*FPRDSAV*APCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          LPOPSAVW(B) = LPOPSAVW(B)+SAVFRAC(B)*FPLPSAV*APCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          RPOPSAVW(B) = RPOPSAVW(B)+SAVFRAC(B)*FPRPSAV*APCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))
          G3PSAVW(B)  =  G3PSAVW(B)+SAVFRAC(B)*FPG3SAV*APCSAV
     $      *((BMLEAF(B)+PLEAF(B)*PRSPSAV(B))
     $      *LEAF(B)+BMSTEM(B)*STEM(B))


          PO4EPI(B) = PO4EPI(B)+SAVFRAC(B)*
     $     (FPIEPI*BMEP(B)+FPIPEP*(PREP(B)+SLSH(B))
     $      -PEP(B))*APCEPI*EP(B)*LEAF(B)*ALAC(J)
          LDOPEPI(B) = LDOPEPI(B)+SAVFRAC(B)*
     $      (FPLDEPI*BMEP(B)+FPLDPEP*(PREP(B)+SLSH(B)))
     $      *APCEPI*EP(B)*LEAF(B)*ALAC(J)
          RDOPEPI(B) = RDOPEPI(B)+SAVFRAC(B)*
     $      (FPRDEPI*BMEP(B)+FPRDPEP*(PREP(B)+SLSH(B)))
     $      *APCEPI*EP(B)*LEAF(B)*ALAC(J)
          LPOPEPI(B) = LPOPEPI(B)+SAVFRAC(B)*
     $      (FPLPEPI*BMEP(B)+FPLPPEP*(PREP(B)+SLSH(B)))
     $      *APCEPI*EP(B)*LEAF(B)*ALAC(J)
          RPOPEPI(B) = RPOPEPI(B)+SAVFRAC(B)*
     $      (FPRPEPI*BMEP(B)+FPRPPEP*(PREP(B)+SLSH(B)))
     $      *APCEPI*EP(B)*LEAF(B)*ALAC(J)
          G3PEPI(B)  =  G3PEPI(B)+SAVFRAC(B)*
     $      (FPG3EPI*BMEP(B)+FPG3PEP*(PREP(B)+SLSH(B)))
     $      *APCEPI*EP(B)*LEAF(B)*ALAC(J)


      END DO

C INTERACTIONS WITH SEDIMENTS

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)
	J = DOMSP(B)
      
C DISSOLVED OXYGEN AND CARBON
  
          SEDDOSAV(B) = SEDDOSAV(B)+SAVFRAC(B)*
     $      (LEAF(B)*AOCR*FDOSR(J)*PLEAF(B)-
     $      (ROOT(B)*BMROOT(B)+TUBER(B)*BMTUBER(B))*AOCR*FDOSAV)
          SEDCSAV(B) = SEDCSAV(B)+SAVFRAC(B)*
     $      (ROOT(B)*BMROOT(B)+TUBER(B)*BMTUBER(B))*
     $      (1.-FDOSAV) 

C NITROGEN AND PHOSPHORUS

          SEDNH4SAV(B) = SEDNH4SAV(B)+SAVFRAC(B)*
     $      FNSEDSAV(B)*PLEAF(B)*ANCSAV*LEAF(B)
          SEDPO4SAV(B) = SEDPO4SAV(B)+SAVFRAC(B)*
     $      FPSEDSAV(B)*PLEAF(B)*APCSAV*LEAF(B)
          SEDNSAV(B)   = SEDNSAV(B)+SAVFRAC(B)*
     $      (ROOT(B)*BMROOT(B)+TUBER(B)*BMTUBER(B))*ANCSAV
          SEDPSAV(B)   = SEDPSAV(B)+SAVFRAC(B)*
     $      (ROOT(B)*BMROOT(B)+TUBER(B)*BMTUBER(B))*APCSAV

      END DO

C COMPUTE NEW SHOOT, ROOT, AND EPIPHYTE BIOMASS

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)
	J = DOMSP(B)

          LFOLD = LEAF(B)
          DELLF = (PLEAF(B)*(1.-PRSPSAV(B))*FPLEAF(JD,J)
     $      -BMLEAF(B))*LEAF(B) + TRTBRLF(JD,J)*TUBER(B)
          ROOT(B) = ROOT(B) + DLTDY*FPROOT(JD,J)*PLEAF(B)*
     $      (1.-PRSPSAV(B))*LEAF(B) - DLTDY*BMROOT(B)*ROOT(B)
          STEM(B) = STEM(B) + DLTDY*FPSTEM(JD,J)*PLEAF(B)*
     $      (1.-PRSPSAV(B))*LEAF(B) - DLTDY*BMSTEM(B)*STEM(B)
          LEAF(B) = LEAF(B) + DLTDY*DELLF
          TUBER(B) = TUBER(B) + DLTDY*FPTUBER(JD,J)*PLEAF(B)*
     $      (1.-PRSPSAV(B))*LFOLD
     $       - DLTDY*(BMTUBER(B)+TRTBRLF(JD,J))*TUBER(B)
          EP(B) = EP(B)*LFOLD*(1.+DLTDY*(PEP(B)-BMEP(B)
     $      -PREP(B)-SLSH(B)))/(LEAF(B)+1.0E-10)

      END DO

C TEMPORARILY SET MINIMUM BIOMASS

cvjp modified 11/3/2005
      DO I=1,NSAVCELL
        B = SAVCELL(I)
        ROOT(B) = MAX(ROOT(B),0.01)
        LEAF(B) = MAX(LEAF(B),0.01)
        STEM(B) = MAX(STEM(B),0.01)
        TUBER(B) = MAX(TUBER(B),0.01)
        EP(B) = MAX(EP(B),0.01)
      END DO
      
C COMPUTE AVERAGE DENSITY AND FRACTION DEPTH OCCUPIED FOR SEDIMENT TRANSPORT

      IF (SEDKIN .NE. 'SSI' .AND. SEDTR_CALC) THEN   
        DO I=1,NSAVCELL
          B = SAVCELL(I)
          J = DOMSP(B)
          HFRAC(B) = 0.0
          AVGDEN(B) = 0.0
          ZMEAN  = SAVDPH(B)  
          HCAN = ACAN(J) + BCAN(J)*(LEAF(B)+STEM(B)) 
          HFRAC(B) = HFRAC(B)+MIN(HCAN/ZMEAN,1.0)*SAVAREA(B)
          HFRAC(B) = HFRAC(B)/(SFA(B)+1.0E-6)
C         HFRAC(B) = HFRAC(B)/(SAVAREA(B)+1.0E-6)
          AVGDEN(B) = LEAF(B)+STEM(B) 
C         AVGDEN(B) = AVGDEN(B)/(SFA(B)+1.0E-6)

          IF (FRDEPTH .EQ. ' FRDEPTH') THEN
C SAV EFFECT BASED ON FRACTION DEPTH OCCUPIED
            SAVEFCT(B) = 0.01 + 0.99*(1.0-HFRAC(B))
          ELSE
C SAV EFFECT BASED ON DENSITY
            SAVEFCT(B) = EXP(-SAVDF(J)*AVGDEN(B))
          END IF
C SAV EFFECTS ON SETTLING VELOCITY
          IF (AVGDEN(B) .LE. DTHRESH(J)) THEN
             SAVSEFCT(B) = 1.0
          ELSE
             SAVSEFCT(B) = EFSAVSTL(J)
          END IF
        END DO
      END IF

      RETURN
      END



