      MODULE roms_init
!
!==================================================== John C. Warner ===
!  Copyright (c) 2005 ROMS/TOMS Group       Alexander F. Shchepetkin   !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This  routine computes the sediment sources and sinks and adds      !
!  then the global sediment tracer fields. Currently, it includes      !
!  the following:                                                      !
!                                                                      !
!  * Vertical settling of sediment in the water column.                !
!  * Erosive and depositional flux interactions of sediment            !
!    between water column and the bed.                                 !
!  * Transport of multiple grain sizes.                                !
!  * Bed layer stratigraphy.                                           !
!  * Bed morphology.                                                   !
!  * Bedload baesd on Meyer Peter Mueller.                             !
!  * Bedload based on Soulsby combined waves + currents                !
!    (p166 Soulsby 1997)                                               !
!  * Seawater/sediment vertical level distribution:                    !
!                                                                      !
!         W-level  RHO-level                                           !
!                                                                      !
!            N     _________                                           !
!                 |         |                                          !
!                 |    N    |                                          !
!          N-1    |_________|  S                                       !
!                 |         |  E                                       !
!                 |   N-1   |  A                                       !
!            2    |_________|  W                                       !
!                 |         |  A                                       !
!                 |    2    |  T                                       !
!            1    |_________|  E                                       !
!                 |         |  R                                       !
!                 |    1    |                                          !
!            0    |_________|_____ bathymetry                          !
!                 |/////////|                                          !
!                 |    1    |                                          !
!            1    |_________|  S                                       !
!                 |         |  E                                       !
!                 |    2    |  D                                       !
!            2    |_________|  I                                       !
!                 |         |  M                                       !
!                 |  Nbed-1 |  E                                       !
!        Nbed-1   |_________|  N                                       !
!                 |         |  T                                       !
!                 |  Nbed   |                                          !
!         Nbed    |_________|                                          !
!                                                                      !
!=======================================================================
!

      USE WQM; USE mod_kinds; USE mod_scalars
      USE mod_sediment
      implicit none
      PUBLIC  :: init_sediment
      PUBLIC  :: Hz, z_w, NumLevels
!
!  Variable declarations.
!
      ! ng=grid number, tile=tile number
      integer, parameter :: ng = 1, tile = 1
      integer, parameter :: Istr = 1, Iend = 1, Jstr = 1 !, Jend = NSBP 
      integer :: Jend 
      integer, parameter :: LBi = 1, UBi = 1, LBj = 1 !, UBj = NSBP
      integer :: UBj
      integer :: nstp = 1, nnew = 1
      real(r8), allocatable :: pm(:,:)
      real(r8), allocatable :: pn(:,:)
      real(r8), allocatable :: rmask(:,:)
      !real(r8), allocatable :: z_w(:,:,:)
      real(r8), allocatable :: h(:,:)
      real(r8), allocatable :: tracer(:,:,:,:,:) 
      real(r8), allocatable :: bed(:,:,:,:)
      real(r8), allocatable :: bed_frac(:,:,:,:)
      real(r8), allocatable :: bed_mass(:,:,:,:,:)
      real(r8), allocatable :: bottom(:,:,:)
      real(r8), allocatable :: tau_w(:,:)
 


      CONTAINS

      PURE FUNCTION NumLevels( j ) RESULT(vert_levels)
!***********************************************************************
!     returns number of levels in water column
!***********************************************************************
      integer, intent(in) :: j
      integer :: vert_levels, box_number

      vert_levels = 1
      box_number = j

      DO WHILE ( box_number .GT. 0 )
          box_number = BD(box_number)
          IF ( box_number .NE. 0 ) THEN
            vert_levels = vert_levels + 1
          END IF
      END DO
      END FUNCTION NumLevels


      FUNCTION Hz( i, j, k ) RESULT(height)
!***********************************************************************
!     retrieves the height of a box
!***********************************************************************
        USE WQM; USE mod_kinds
        integer, intent(in) :: i, j, k
        integer  :: counter = 0, cbox = 0
        real(r8) :: height
        logical  :: bFound = .FALSE.

        cbox = j
        counter = NumLevels(j)

        DO
      
          IF ( counter == k ) THEN
            height = BL(cbox, 3)
            bFound = .TRUE.
            EXIT
          END IF

          cbox = BD(cbox)

          IF ( cbox == 0 ) THEN
            height = 1E-20
            bFound = .TRUE.
            EXIT
          END IF

          counter = counter - 1

        END DO

!       IF ( bFound == .FALSE. .OR. height == 0.0 ) THEN
        IF ( (.not.(bFound))  .OR. height == 0.0 ) THEN
          height = 1E-20
        END IF

      END FUNCTION Hz


      FUNCTION z_w( i, j, k ) RESULT(height)
!***********************************************************************
!     retrieves the actual height of a box
!***********************************************************************
        integer, intent(in) :: i, j, k
        real(r8) :: height
        
        height = Hz(i,j,k)
      END FUNCTION z_w


      SUBROUTINE init_sediment()
        USE WQM; USE mod_sediment; USE FILE_INFO
        USE mod_scalars
        INTEGER I, J, K, N, ised, box_number, indx
        real(r8) :: box_area
        INTEGER IDCELL,L_19,L
        real SAND,SILT,CLAY,ORGSED,XSED1,XSED2,XSED3,XSED4
        character (len=3) :: BED_INI
        character (len=8) :: SVEL_INI
        character (len=72) :: BED_OF, SVELFN
        real :: THICK(NBED)
        real :: d50val,scd1,scd2,scd12, dratio, bulkini
        integer ksc0, ksc1
      
     !  assign upper limits
        Jend = NSB
        UBj  = NSB
        
     !  allocate arrays
        allocate( pm(UBi,UBj) )
        allocate( pn(UBi,UBj) )
        allocate( rmask(UBi,UBj) )
        !allocate( z_w(UBi,UBj,NLP) )
        allocate( h(UBi,UBj) )
        allocate( tracer(UBi,UBj,NLP,3,NST) )
        allocate( bed(UBi,UBj,Nbed,MBEDP) )
        allocate( bed_frac(UBi,UBj,Nbed,NST) )
        allocate( bed_mass(UBi,UBj,Nbed,2,NST) )
        allocate( bottom(UBi,UBj,MBOTP) )
        allocate( tau_w(Istr-3:Iend+3,Jstr-3:Jend+3) )
        
     !
     !  INITIALIZE TIME
     !
        time(1) = DLT

     !
     !  SET MORPHOLOGY FACTOR
     !
        DO K=1,NST
           morph_fac(K, Ngrids) = 1
        END DO

     !
     !  SET THE rmask
     !
        DO K=1,NSB
           rmask(1, K) = 1
        END DO

     !
     !  CRITERIA FOR NEW LAYER FORMATION
     !
        newlayer_thick( 1 ) = 0.002      ! meters

     !  SET pm AND pn
     !  pm  Coordinate transformation metric "m" (1/meters)         !
     !      associated with the differential distances in XI        !
     !      at RHO-points.                                          !
     !  pn  Coordinate transformation metric "n" (1/meters)         !
     !      associated with the differential distances in ETA.      !
     !      at RHO-points.                                          !
        DO K=1, NSB
           pm(1, K) = 1
           pn(1, K) = 1
        END DO


     !
     !  MEDIAN DIAMETER (meters)
        READ(CON,'(//,8X,4G8.0)') (D50(K),K=1,NSDCLS)  
        Sd50(1, 1) = D50(1) * 1E-6 ! convert meters to microns 
        Sd50(2, 1) = D50(2) * 1E-6
        Sd50(3, 1) = D50(3) * 1E-6
        Sd50(4, 1) = D50(4) * 1E-6
     !
     !  DENSITY (kg/m^3)
        READ(CON,'(//,8X,4G8.0)') (Srho(K,1),K=1,NSDCLS)
     !
     !  SETTLING VELOCITY (u/s)
        READ(CON,'(//,8X,A8)') SVEL_INI
        READ(CON,'(//8x,a72)') SVELFN
        OPEN(11,FILE=SVELFN,STATUS='OLD')
        READ(11,*)            !  TITLE LINE
        IF (SVEL_INI .EQ. ' UNIFORM') THEN
          READ(11,*) IDCELL, WSTLCLY(1), WSTLSLT(1), WSTLSND(1), &   
            WSTLORG(1)
          DO B=2,NB
            WSTLCLY(B) = WSTLCLY(1)
            WSTLSLT(B) = WSTLSLT(1)
            WSTLSND(B) = WSTLSND(1)
            WSTLORG(B) = WSTLORG(1)
          END DO
        ELSE
          DO B=1,NB
             READ(11,*) IDCELL, WSTLCLY(B), WSTLSLT(B), WSTLSND(B), &
              WSTLORG(B)
          END DO
        END IF
        CLOSE(11)
        WSTLCLY(0) = 0
        WSTLSLT(0) = 0
        WSTLSND(0) = 0
        WSTLORG(0) = 0

        DO B=1,NB  !  CONVERT u/S TO M/S
          WSTLCLY(B) = WSTLCLY(B)/1.0E6
          WSTLSLT(B) = WSTLSLT(B)/1.0E6
          WSTLSND(B) = WSTLSND(B)/1.0E6
          WSTLORG(B) = WSTLORG(B)/1.0E6
        END DO  

        DO B=1,NSB ! ASSIGN SETTLING VELOCITIES INTO SEDIMENT FOR ROMS	
          Wsed(1, B) = WSTLCLY(B)
          Wsed(2, B) = WSTLSLT(B)
          Wsed(3, B) = WSTLSND(B)
          Wsed(4, B) = WSTLORG(B)
        END DO

! ASSIGN BASE SETTLING VELOCITIES THAT WILL BE MODIFIED BY SAV EFFECTS	
        DO B=1,NSB
          WSTBCLY(B) = WSTLCLY(B)
          WSTBSLT(B) = WSTLSLT(B)
          WSTBORG(B) = WSTLORG(B)
        END DO
      
     !  CRITICAL SHEAR STRESS (N/m^2)
        READ(CON,'(//,8X,4G8.0)') (tau_ce(K,1),K=1,NSDCLS)
      
     !  FOR ROMS, CONVERT FROM PASCAL TO M**2/S**2  CFC
        DO K=1,NSDCLS
           TAU_CE(K,1) = TAU_CE(K,1)/RHO0
        END DO
      
     !  EROSION RATE (kg/m^2/s/P for Sanford's formula), Gamma0 dimensionless
        READ(CON,'(//,8X,5G8.0)') (Erate(K,1),K=1,NSDCLS), Gamma0 ! g/m^2/s/P
        DO K=1,NSDCLS
          Erate(K, 1) = Erate(K,1)*1.e-3 ! kg/m^2/s/P
        END DO

!
!  see if bed initial condition is constant
!
        READ(CON,'(//,13X,A3)') BED_INI
        READ (CON,'(//8x,a72)') BEDFN
        READ (CON,'(//8x,a72)') BED_OF
        open(151,file=bed_of,form='unformatted',status='unknown')

!
!  ASSIGN CLASS PERCENTAGE TO ALL THE BOTTOM CELLS
!
      if(BED_INI.eq.' ON') then
        OPEN(11,FILE=BEDFN,FORM='UNFORMATTED',STATUS='OLD')
        read(11) (((bed_frac(1,J,L,K),K=1,NSDCLS),L=1,NBED),J=1,NSB)
        read(11) ((bed(1,J,L,ithck),L=1,NBED),J=1,NSB)
        read(11) ((bed(1,J,L,iporo),L=1,NBED),J=1,NSB)
        read(11) ((bed(1,J,L,iaged),L=1,NBED),J=1,NSB)
        read(11) ((bed(1,J,L,idiff),L=1,NBED),J=1,NSB)
        close(11)
        DO J=1,NSB
          DO L=1,NBED
            DO K=1,NSDCLS
              bed_mass(1,J,L,1,K) = Srho(K,1) * bed_frac(1,J,L,K) * & 
     &          bed(1,J,L,ithck)  * (1.0_r8 - bed(1,J,L,iporo))  ! kg/m^2, porosity added by CFC 
            END DO
          END DO
        END DO
        READ(CON,'(//,8X,10G8.0)') (THICK(L),L=1,LAYMAX)
      else if(BED_INI.eq.'OFF') then
        OPEN(11,FILE=BEDFN,STATUS='OLD')
        READ(11,*)
        DO B=1,NSB ! ROMS bed fractions by volume, not mass CFC
          READ(11,*) IDCELL,L_19,CLAY,SILT,SAND,ORGSED
          XSED1=0.01*CLAY
          XSED2=0.01*SILT
          XSED3=0.01*SAND
          XSED4=1.-XSED1-xsed2-xsed3
          DO L=1,NBED
            ! These are by volume.  Converted to mass fraction in sediment_from_ROMS.F90
            bed_frac(1,B,L,1) = xsed1
            bed_frac(1,B,L,2) = xsed2
            bed_frac(1,B,L,3) = xsed3
            bed_frac(1,B,L,4) = xsed4
          END DO
        END DO
        CLOSE(11)

        !  READ IN INITIAL BED THICKNESS (CM)
        READ(CON,'(//,8X,10G8.0)') (THICK(L),L=1,LAYMAX)

        !  SET BED LAYER PROPERTIES
        DO J=1,NSB
          DO L=1,NBED
            bed(1,J,L,ithck)=.01*THICK(L)
            bed(1,J,L,iporo)=.8
            bed(1,J,L,iaged) = 0   ! seconds
            bed(1,J,L,idiff) = 0
            DO K=1,NSDCLS
              bed_mass(1,J,L,1,K) = Srho(K,1) * bed_frac(1,J,L,K) * &
     &          bed(1,J,L,ithck) * (1.0_r8 - bed(1,J,L,iporo))  ! kg/m^2, porosity added by CFC
            END DO
          END DO
        END DO
        
      endif
      
      !  BOTTOM PROPERTIES (TOP LAYER)
      DO J=1, NSB
        bottom(1,j,isd50) = 0
        bottom(1,j,idens) = 0
        bottom(1,j,iwsed) = 0
        bottom(1,j,itauc) = 0

        do k=1,nsdcls
        bottom(1,j,isd50) = bottom(1,j,isd50)+bed_frac(1,j,1,k)*Sd50(k, 1)
        bottom(1,j,idens) = bottom(1,j,idens)+bed_frac(1,j,1,k)*Srho(k, 1)
        bottom(1,j,iwsed) = bottom(1,j,iwsed)+bed_frac(1,j,1,k)*Wsed(k, 1)
        bottom(1,j,itauc) = bottom(1,j,itauc)+bed_frac(1,j,1,k)*tau_ce(k, 1)
        end do

        bottom(1,j,irlen) = 0
        bottom(1,j,irhgt) = 0
        bottom(1,j,ibwav) = 0
        bottom(1,j,izNik) = 0
        bottom(1,j,izbio) = 0
        bottom(1,j,izbfm) = 0
        bottom(1,j,izbld) = 0
        bottom(1,j,izapp) = 0
        bottom(1,j,izwbl) = 0
        bottom(1,j,izdef) = 0
        bottom(1,j,iactv) = bottom(1,j,isd50) * 8    ! m
        bottom(1,j,ishgt) = 0
      END DO

      END SUBROUTINE init_sediment


      END MODULE roms_init
