

      MODULE sediment_mod
!
!==================================================== John C. Warner ===
!  Copyright (c) 2005 ROMS/TOMS Group       Alexander F. Shchepetkin   !
!================================================== Hernan G. Arango ===
!                                                                      !
!  This  routine computes the sediment sources and sinks and adds      !
!  then the global sediment tracer fields. Currently, it includes      !
!  the following:                                                      !
!                                                                      !
!  * Vertical settling of sediment in the water column.                !
!  * Erosive and depositional flux interactions of sediment            !
!    between water column and the bed.                                 !
!  * Transport of multiple grain sizes.                                !
!  * Bed layer stratigraphy.                                           !
!  * Bed morphology.                                                   !
!  * Bedload baesd on Meyer Peter Mueller.                             !
!  * Bedload based on Soulsby combined waves + currents                !
!    (p166 Soulsby 1997)                                               !
!  * Seawater/sediment vertical level distribution:                    !
!                                                                      !
!         W-level  RHO-level                                           !
!                                                                      !
!            N     _________                                           !
!                 |         |                                          !
!                 |    N    |                                          !
!          N-1    |_________|  S                                       !
!                 |         |  E                                       !
!                 |   N-1   |  A                                       !
!            2    |_________|  W                                       !
!                 |         |  A                                       !
!                 |    2    |  T                                       !
!            1    |_________|  E                                       !
!                 |         |  R                                       !
!                 |    1    |                                          !
!            0    |_________|_____ bathymetry                          !
!                 |/////////|                                          !
!                 |    1    |                                          !
!            1    |_________|  S                                       !
!                 |         |  E                                       !
!                 |    2    |  D                                       !
!            2    |_________|  I                                       !
!                 |         |  M                                       !
!                 |  Nbed-1 |  E                                       !
!        Nbed-1   |_________|  N                                       !
!                 |         |  T                                       !
!                 |  Nbed   |                                          !
!         Nbed    |_________|                                          !
!                                                                      !
!=======================================================================
!

      USE WQM; USE mod_kinds; USE mod_scalars
      USE roms_init; USE mod_sediment

      implicit none
      PUBLIC  :: sediment

      CONTAINS

!
!***********************************************************************
      SUBROUTINE sediment ()
!***********************************************************************
!

!
!  Local variable declarations.
!
      integer :: box_number, box_counter
      real(r8) :: box_volume

      integer :: IstrR, IendR, JstrR, JendR
      integer :: Ksed, i, indx, ised, j, k, ks
      integer :: bnew
      real(r8), parameter :: eps = 1.0E-10_r8
      real(r8) :: cff, cff1, cff2, cff3, cffL, cffR, dltL, dltR
      real(r8) :: cu, ero_flux, cff4
      real(r8) :: thck_avail, thck_to_add
      integer, dimension(Istr-3:Iend+3,NLP) :: ksource
      real(r8), dimension(Istr-3:Iend+3,0:NLP) :: FC
      real(r8), dimension(Istr-3:Iend+3,NLP) :: Hz_inv
      real(r8), dimension(Istr-3:Iend+3,NLP) :: Hz_inv2
      real(r8), dimension(Istr-3:Iend+3,NLP) :: Hz_inv3
      real(r8), dimension(Istr-3:Iend+3,NLP) :: qc
      real(r8), dimension(Istr-3:Iend+3,NLP) :: qR
      real(r8), dimension(Istr-3:Iend+3,NLP) :: qL
      real(r8), dimension(Istr-3:Iend+3,NLP) :: WR
      real(r8), dimension(Istr-3:Iend+3,NLP) :: WL
      real(r8), dimension(Istr-3:Iend+3,NST) :: dep_mass

!
!  sk
!      
      real(r8):: tau_crit
      real(r8):: Cbee, Ssfm, Ca

      bnew = nstp

    ! set the time step
      dts(1) = DLT

     !  SET THE INITIAL CONCENTRATION IN EACH BOX
        DO J=1,NSB
          box_number = BBN(J)
          DO K=1,NumLevels(J)
            tracer(1,J,K,nnew,idsed(1)) = SEDCLY(box_number) / 1000 * Hz(1,J,K)  ! g/m^3 * m / 1000 = kg/m^2
            tracer(1,J,K,nnew,idsed(2)) = SEDSLT(box_number) / 1000 * Hz(1,J,K)  ! g/m^3 * m / 1000 = kg/m^2
            tracer(1,J,K,nnew,idsed(3)) = SEDSND(box_number) / 1000 * Hz(1,J,K)  ! g/m^3 * m / 1000 = kg/m^2
            tracer(1,J,K,nnew,idsed(4)) = SEDORG(box_number) / 1000 * Hz(1,J,K)  ! g/m^3 * m / 1000 = kg/m^2
            box_number = BU(box_number)
          END DO
        END DO

     !
     !  BOTTOM STRESS (m**2/s**2) FOR ROMS
     !
        DO J=1,NSB
          tau_w(1,J) = USTARC(J)**2./10000.  ! CONVERT CM/S TO M**2/S**2 CFC
        END DO

        IF (SAV_CALC .AND. SEDKIN .NE. 'SSI') THEN
          DO I=1,NSAVCELL
            J = SAVCELL(I)
            tau_w(1,J) = tau_w(1,J)*SAVEFCT(J)
            Wsed(1,J) = WSTBCLY(J)*SAVSEFCT(J)
            Wsed(2,J) = WSTBSLT(J)*SAVSEFCT(J)
            Wsed(4,J) = WSTBORG(J)*SAVSEFCT(J)
          END DO
        END IF

!       Sea Carousel
!        do i=1,12
!	  if (jday .gt. 0.01444*float(i-1) .and. jday .lt. &
!	    0.01444*float(i)) then
!	    do j=1,nsb
!	      tau_w(1,j) = (0.035*float(i-1)+1.0e-6)/rho0
!	    end do
!	  end if
!	end do	

!
!  Update mean surface properties.
!  Sd50 must be positive definite, due to BBL routines.
!  Srho must be >1000, due to (s-1) in BBL routines.
!
      DO j=Jstr,Jend
        DO i=Istr,Iend
          cff3=0.0_r8
          DO ised=1,NST
            cff3=cff3+bed_mass(i,j,1,nnew,ised)
          END DO
          DO ised=1,NST
            bed_frac(i,j,1,ised)=bed_mass(i,j,1,nnew,ised)/             &
     &                           MAX(cff3,eps)
          END DO
!
          cff1=1.0_r8
          cff2=1.0_r8
          cff3=1.0_r8
          cff4=1.0_r8
          DO ised=1,NST
            cff1=cff1*tau_ce(ised,ng)**bed_frac(i,j,1,ised)
            cff2=cff2*Sd50(ised,ng)**bed_frac(i,j,1,ised)
            cff3=cff3*(Wsed(ised,j)+eps)**bed_frac(i,j,1,ised)
            cff4=cff4*Srho(ised,ng)**bed_frac(i,j,1,ised)
          END DO
          bottom(i,j,itauc)=cff1
          bottom(i,j,isd50)=cff2
          bottom(i,j,iwsed)=cff3
          bottom(i,j,idens)=MAX(cff4,1050.0_r8)
        END DO
      END DO

!
!-----------------------------------------------------------------------
!  Add sediment Source/Sink terms.
!-----------------------------------------------------------------------
!
!  Compute inverse thicknesses to avoid repeated divisions.
!
      J_LOOP : DO j=Jstr,Jend
        DO k=1,NumLevels(j)
          DO i=Istr,Iend
            Hz_inv(i,k)=1.0_r8/Hz(i,j,k)
          END DO
        END DO
        DO k=1,NumLevels(j)-1
          DO i=Istr,Iend
            Hz_inv2(i,k)=1.0_r8/(Hz(i,j,k)+Hz(i,j,k+1))
          END DO
        END DO
        DO k=2,NumLevels(j)-1
          DO i=Istr,Iend
            Hz_inv3(i,k)=1.0_r8/(Hz(i,j,k-1)+Hz(i,j,k)+Hz(i,j,k+1))
          END DO
        END DO
!
!  Copy concentration of suspended sediment into scratch array "qc"
!  (q-central, restrict it to be positive) which is hereafter
!  interpreted as a set of grid-box averaged values for sediment
!  concentration.
!
        SED_LOOP: DO ised=1,NST
          indx=idsed(ised)
          DO k=1,NumLevels(j)
            DO i=Istr,Iend
              qc(i,k)=tracer(i,j,k,nnew,indx)*Hz_inv(i,k)
              IF ( k == 1 ) THEN   ! changed to 1 from NumLevels(j) by CFC
                FC(i,0) = min(Wsed(ised,j),1./Hz_inv(i,k)/dts(1)) &
    &             * qc(i,k) * dts(1)
                if (sfeeder .and. sedkin .ne. 'SSI') &     
                   FC(i,0) = FC(i,0)+sf_fc(i,j)*dts(1)   ! suspension feeders
              END IF
            END DO
          END DO

          DO i=Istr,Iend

            tau_crit = tau_ce(ised,ng) 
            cff=1.0_r8/tau_crit
            dep_mass(i,ised)=0.0_r8
!
!  Compute erosion, ero_flux (kg/m2).
!
            if (ised .le. 2 .or. ised .eq. 4) then  ! clay or silt
              cff1=(1.0_r8-bed(i,j,1,iporo))*bed_frac(i,j,1,ised)
! CFC's attempt to make this equivalent to Sanford          
              cff2=dts(ng)*Erate(ised,ng)*rho0*tau_crit               &
     &        *bed_frac(i,j,1,ised)                                   &
     &        /(bed_frac(i,j,1,1)+bed_frac(i,j,1,2)                   &
     &        +bed_frac(i,j,1,4)+1.0e-6)  
              cff3=Srho(ised,ng)*cff1
              cff4=bed_mass(i,j,1,bnew,ised)
              cff3=cff3/morph_fac(ised,ng)
              cff4=cff4/morph_fac(ised,ng)
              ero_flux=MIN(MAX(0.0_r8,cff2*(cff*tau_w(i,j)-1.0_r8)),    &
     &                   MIN(cff3*bottom(i,j,iactv),                    &
     &                   cff4)+FC(i,0))
            else   !  sand
              cff1=(1.0_r8-bed(i,j,1,iporo))*bed_frac(i,j,1,ised)
              if (Wsed(ised,j)/(0.4_r8 * sqrt(tau_w(i,j)+1.0e-6)) .le.  &
     &          2.5) then
                Cbee = 1.0_r8 - bed(i,j,1,iporo)
                Ssfm = max(0.0_r8,cff * tau_w(i,j) - 1.0_r8)
                Ca = bed_frac(i,j,1,ised) * Cbee * Gamma0 * Ssfm / &
     &            (1.0_r8 + Gamma0 * Ssfm)
              else
                Ca = 0.0
              end if
              cff2=dts(ng)*Srho(ised,ng)*Wsed(ised,j)*Ca 
              cff3=Srho(ised,ng)*cff1
              cff4=bed_mass(i,j,1,bnew,ised)
              cff3=cff3/morph_fac(ised,ng)
              cff4=cff4/morph_fac(ised,ng)
              ero_flux=MIN(MAX(0.0_r8,cff2),                  &
     &          MIN(cff3*bottom(i,j,iactv), cff4)+FC(i,0))
           end if

!
!  Compute new sediment concentrations.
!
            qc(i,1)=qc(i,1)+ero_flux*Hz_inv(i,1)
!
!
! Apply morphology factor.
!
          ero_flux=ero_flux*morph_fac(ised,ng)
          FC(i,0)=FC(i,0)*morph_fac(ised,ng)

!
!  Check if depositional.
!
            IF ((ero_flux-FC(i,0)).lt.0.0_r8) THEN
!
!  If first time step of deposit, then store deposit material in
!  temporary array, dep_mass.
!
              !PRINT *, time(ng), bed(i,j,1,iaged)+1.1_r8*dts(ng), bed(i,j,1,ithck), newlayer_thick(ng)
              
              !IF ((time(ng).gt.(bed(i,j,1,iaged)+1.1_r8*dts(ng))).and.   &
              !    (bed(i,j,1,ithck).gt.newlayer_thick(ng)))THEN
!                  (bed(i,j,1,ithck).gt.0.005_r8))THEN
              IF ( bed(i,j,1,ithck) .gt. newlayer_thick(ng) ) THEN
                dep_mass(i,ised)=-(ero_flux-FC(i,0))+eps
              ELSE
!
!  If it is not first time step of deposit, update bed thickness of
!  top layer.
!  
                bed(i,j,1,ithck)=MAX(bed(i,j,1,ithck)-                  &
     &                               (ero_flux-FC(i,0))/                &
     &                               (Srho(ised,ng)*                    &
     &                               (1.0_r8-bed(i,j,1,iporo))),0.0_r8)
                bed(i,j,1,ithck)=bed(i,j,1,ithck)*rmask(i,j)
!
!  Upate bed mass of top layer.
!
                bed_mass(i,j,1,nnew,ised)=MAX(bed_mass(i,j,1,bnew,ised)-&
     &                                   (ero_flux-FC(i,0)),0.0_r8)
                bed(i,j,1,iaged)=time(ng)
              END IF
!
!  Else, if erosional.
!
            ELSE
!
!  Update bed thickness of top layer.
!  
              bed(i,j,1,ithck)=MAX(bed(i,j,1,ithck)-                    &
     &                             (ero_flux-FC(i,0))/                  &
     &                             (Srho(ised,ng)*                      &
     &                             (1.0_r8-bed(i,j,1,iporo))),0.0_r8)
              bed(i,j,1,ithck)=bed(i,j,1,ithck)*rmask(i,j)
!
!  Upate bed mass of top layer.
!
              bed_mass(i,j,1,nnew,ised)=MAX(bed_mass(i,j,1,bnew,ised)-  &
     &                                      (ero_flux-FC(i,0)),0.0_r8)

            END IF
!
!  Update bed mass arrays.
!
            DO k=2,Nbed
              bed_mass(i,j,k,nnew,ised)=bed_mass(i,j,k,nstp,ised)
            END DO
          END DO
!
!-----------------------------------------------------------------------
!  Update global tracer variables (m Tunits).
!-----------------------------------------------------------------------
!
          DO k=1,NumLevels(j)
            DO i=Istr,Iend
              tracer(i,j,k,nnew,indx)=Hz(i,j,k)*qc(i,k)
              tracer(i,j,k,3,indx)=Hz(i,j,k)*qc(i,k)
            END DO
          END DO
        END DO SED_LOOP

!
!  Upate bed fraction of top layer.
!
        DO i=Istr,Iend
          cff3=0.0_r8
          DO ised=1,NST
             cff3=cff3+bed_mass(i,j,1,nnew,ised)
          END DO
          DO ised=1,NST
            bed_frac(i,j,1,ised)=bed_mass(i,j,1,nnew,ised)/MAX(cff3,eps)
          END DO
        END DO
!
!  If first time step of deposit, create new layer and combine bottom
!  two bed layers.
!
          DO i=Istr,Iend
            cff=0.0_r8
!
!  Determine if deposition ocurred here.
!
            DO ised=1,NST
               cff=cff+dep_mass(i,ised)
            END DO
            IF (cff.gt.0.0_r8) THEN
              IF (NBED.gt.1) THEN
!
!  Combine bottom layers.
!
                bed(i,j,Nbed,ithck)=bed(i,j,Nbed-1,ithck)+              &
                                    bed(i,j,Nbed  ,ithck)
                bed(i,j,Nbed,iporo)=0.5_r8*(bed(i,j,Nbed-1,iporo)+      &
     &                                      bed(i,j,Nbed,iporo))
                bed(i,j,Nbed,iaged)=0.5_r8*(bed(i,j,Nbed-1,iaged)+      &
     &                                      bed(i,j,Nbed,iaged))
                cff3=0.0_r8
                DO ised=1,NST
                  bed_mass(i,j,Nbed,nnew,ised)=                         &
     &                               bed_mass(i,j,Nbed-1,nnew,ised)+    &
     &                               bed_mass(i,j,Nbed  ,nnew,ised)
                  cff3=cff3+bed_mass(i,j,Nbed,nnew,ised)
                END DO
                cff3=1.0_r8/MAX(cff3,eps)
                DO ised=1,NST
                  bed_frac(i,j,Nbed,ised)=bed_mass(i,j,Nbed,nnew,ised)* &
     &                                    cff3
                END DO
!
!  Push layers down.
!
                DO k=Nbed-1,2,-1
                  bed(i,j,k,ithck)=bed(i,j,k-1,ithck)
                  bed(i,j,k,iporo)=bed(i,j,k-1,iporo)
                  bed(i,j,k,iaged)=bed(i,j,k-1,iaged)
                  DO ised =1,NST
                    bed_frac(i,j,k,ised)=bed_frac(i,j,k-1,ised)
                    bed_mass(i,j,k,nnew,ised)=bed_mass(i,j,k-1,nnew,ised)
                  END DO
                END DO
!
!  Set new top layer parameters.
!
                bed(i,j,1,ithck)=0.0_r8
                DO ised=1,NST
                  bed_mass(i,j,1,nnew,ised)=0.0_r8
                END DO
              END IF !NBED=1
             cff3=0.0_r8
              DO ised=1,NST
                bed_mass(i,j,1,nnew,ised)=bed_mass(i,j,1,nnew,ised)+    &
     &                                    dep_mass(i,ised)
                cff3=cff3+bed_mass(i,j,1,nnew,ised)
                bed(i,j,1,ithck)=bed(i,j,1,ithck)+                      &
     &                           dep_mass(i,ised)/                      &
     &                           (Srho(ised,ng)*                        &
     &                            (1.0_r8-bed(i,j,1,iporo)))
              END DO
              bed(i,j,1,iaged)=time(ng)
              DO ised=1,NST
                bed_frac(i,j,1,ised)=bed_mass(i,j,1,nnew,ised)/         &
     &                               MAX(cff3,eps)
!
!  Reset deposition mass.
!
                dep_mass(i,ised)=0.0_r8
              END DO
            END IF
!
          END DO
      END DO J_LOOP
!
!  End of Suspended Sediment only section.
!
!  Determine top layer to be at least active layer thickness.
!
!  Ensure top bed layer thickness is greater or equal than active layer
!  thickness. If need to add sed to top layer, then entrain from lower
!  levels. Create new layers at bottom to maintain Nbed.
!
      J_LOOP2 : DO j=Jstr,Jend
        DO i=Istr,Iend
!
!  Calculate active layer thickness, bottom(i,j,iactv).
!
          bottom(i,j,iactv)=MAX(0.0_r8,                                 &
     &                          0.007_r8*                               &
     &                          (tau_w(i,j)-bottom(i,j,itauc))*rho0)+   &
     &                          6.0_r8*bottom(i,j,isd50)
!
!
! Apply morphology factor.
!
          DO ised=1,NST
            bottom(i,j,iactv)=bottom(i,j,iactv)*morph_fac(ised,ng)
          END DO
!
          IF (bottom(i,j,iactv).gt.bed(i,j,1,ithck)) THEN
            IF (Nbed.eq.1) THEN
              bottom(i,j,iactv)=bed(i,j,1,ithck)
            ELSE
              thck_to_add=bottom(i,j,iactv)-bed(i,j,1,ithck)
              thck_avail=0.0_r8
              Ksed=1                                        ! initialize
              DO k=2,Nbed
                IF (thck_avail.lt.thck_to_add) THEN
                  thck_avail=thck_avail+bed(i,j,k,ithck)
                  Ksed=k
                END IF
              END DO
!
!  Catch here if there was not enough bed material.
!
              IF (thck_avail.lt.thck_to_add) THEN
                bottom(i,j,iactv)=bed(i,j,1,ithck)+thck_avail
                thck_to_add=thck_avail
              END IF
!
!  Update bed mass of top layer and fractional layer.
!
              cff2=MAX(thck_avail-thck_to_add,0.0_r8)/                  &
     &             MAX(bed(i,j,Ksed,ithck),eps)
              DO ised=1,NST
                cff1=0.0_r8
                DO k=1,Ksed
                  cff1=cff1+bed_mass(i,j,k,nnew,ised)
                END DO
                bed_mass(i,j,1   ,nnew,ised)=cff1-                      &
     &                                bed_mass(i,j,Ksed,nnew,ised)*cff2
                bed_mass(i,j,Ksed,nnew,ised)=                           &
     &                                bed_mass(i,j,Ksed,nnew,ised)*cff2
              END DO
!
!  Update thickness of fractional layer ksource_sed.
!
              bed(i,j,Ksed,ithck)=MAX(thck_avail-thck_to_add,0.0_r8)
!
!  Upate bed fraction of top layer.
!
              cff3=0.0_r8
              DO ised=1,NST
                cff3=cff3+bed_mass(i,j,1,nnew,ised)
              END DO
              DO ised=1,NST
                bed_frac(i,j,1,ised)=bed_mass(i,j,1,nnew,ised)/         &
     &                               MAX(cff3,eps)
              END DO
!
!  Upate bed thickness of top layer.
!
              bed(i,j,1,ithck)=bottom(i,j,iactv)
!
!  Pull all layers closer to the surface.
!
              DO k=Ksed,Nbed
                ks=Ksed-2
                bed(i,j,k-ks,ithck)=bed(i,j,k,ithck)
                bed(i,j,k-ks,iporo)=bed(i,j,k,iporo)
                bed(i,j,k-ks,iaged)=bed(i,j,k,iaged)
                DO ised=1,NST
                  bed_frac(i,j,k-ks,ised)=bed_frac(i,j,k,ised)
                  bed_mass(i,j,k-ks,nnew,ised)=bed_mass(i,j,k,nnew,ised)
                END DO
              END DO
!
!  Add new layers onto the bottom. Split what was in the bottom layer to
!  fill these new empty cells. ("ks" is the number of new layers).
!
              ks=Ksed-2
              cff=1.0_r8/REAL(ks+1,r8)
              DO k=Nbed,Nbed-ks,-1
                bed(i,j,k,ithck)=bed(i,j,Nbed-ks,ithck)*cff
                bed(i,j,k,iaged)=bed(i,j,Nbed-ks,iaged)
                DO ised=1,NST
                  bed_frac(i,j,k,ised)=bed_frac(i,j,Nbed-ks,ised)
                  bed_mass(i,j,k,nnew,ised)=                            &
     &                             bed_mass(i,j,Nbed-ks,nnew,ised)*cff
                END DO
              END DO
            END IF  ! Nbed > 1
          END IF  ! increase top bed layer
!
!  Update mean surface properties.
!  Sd50 must be positive definite, due to BBL routines.
!  Srho must be >1000, due to (s-1) in BBL routines
!
          cff1=1.0_r8
          cff2=1.0_r8
          cff3=1.0_r8
          cff4=1.0_r8
          DO ised=1,NST
            cff1=cff1*tau_ce(ised,ng)**bed_frac(i,j,1,ised)
            cff2=cff2*Sd50(ised,ng)**bed_frac(i,j,1,ised)
            cff3=cff3*(Wsed(ised,j)+eps)**bed_frac(i,j,1,ised)
            cff4=cff4*Srho(ised,ng)**bed_frac(i,j,1,ised)
          END DO
          bottom(i,j,itauc)=cff1
          bottom(i,j,isd50)=cff2
          bottom(i,j,iwsed)=cff3
          bottom(i,j,idens)=MAX(cff4,1050.0_r8)
        END DO
      END DO J_LOOP2
      
!  update current time

      time(1) = time(1) + DLT

!  update ICM suspended solids data
!***********************************************************************
      
      ! loop through surface boxes
      DO j=Jstr,Jend
      
        ! dummy loop, i always equals 1
        DO i=Istr,Iend
      
          ! loop through levels
          DO k=1,NumLevels(j)

            ! get box number
            box_number = BBN(j)
            DO box_counter=2,k
              box_number = BU(box_number)
            END DO
            ! get change in mass ( g/m^3/s )
            DTSCLY(box_number) = &
     &        (tracer(i,j,k,1,1) / Hz(1,j,k) * 1000 - SEDCLY(box_number)) / dts(1) &
     &        + DTSCLY(box_number)
            DTSSLT(box_number) = (tracer(i,j,k,1,2) / Hz(1,j,k) * 1000 &
     &        - SEDSLT(box_number)) / dts(1) + DTSSLT(box_number)
            DTSSND(box_number) = &
     &        (tracer(i,j,k,1,3) / Hz(1,j,k) * 1000 - SEDSND(box_number)) / dts(1) &
     &         + DTSSND(box_number)
            DTSORG(box_number) = &
     &        (tracer(i,j,k,1,4) / Hz(1,j,k) * 1000 - SEDORG(box_number)) / dts(1) &
     &         + DTSORG(box_number)

      
          END DO
      
        END DO
      
      END DO

      !PRINT *, SEDCLY(1), SEDSLT(1)
      !PRINT *, SEDCLY(11), SEDSLT(11)
      !PRINT *, SEDCLY(21), SEDSLT(21)
      !Print *, ''
      !PRINT *, (bed_mass(1,1,K,1,2),K=1,NBED)
      !Print *, ''
      !PRINT *, Jday,DTSCLY(21), DTSSND(21), DTSCLY(22), DTSSND(22)
      !PRINT *, Jday,SEDCLY(21), SEDSND(21), SEDCLY(22), SEDSND(22)
      
      END SUBROUTINE sediment
      
      END MODULE sediment_mod


!-------------------------------------------------------------------
!  MAKE SEDIMENT SETTLE IN A BOX AT EACH TIME STEP
!  ( This routine lifted from wqm_sedzl.F )
!  vjp 02/7/2011
!-------------------------------------------------------------------
      SUBROUTINE SED_STL
      USE WQM; USE FILE_INFO
      INTEGER SB,SURFACE,ACTLAYER(NSBP)
      REAL ELAY(NSDCLS)
      INTEGER F
      DATA XKAPPA/.4/

! SAV EFFECT ON SETTLING VELOCITY
       IF (SAV_CALC .AND. SEDKIN .NE. 'SSI') THEN
          DO I=1,NSAVCELL
            J = SAVCELL(I)
            WSTLCLY(J) = WSTBCLY(J)*SAVSEFCT(J)
            WSTLSLT(J) = WSTBSLT(J)*SAVSEFCT(J)
            WSTLORG(J) = WSTBORG(J)*SAVSEFCT(J)
          END DO
       END IF

! FIRST SURFACE BOXES (BU(B) = 0)
      DO B=1,NSB
        DTSS1= -WSTLCLY(B)*SEDCLY(B) ! G/M^2/S
        DTSS2= -MIN(WSTLSLT(B),BL(B,3)/DLT)*SEDSLT(B)
        DTSS3= -MIN(WSTLSND(B),BL(B,3)/DLT)*SEDSND(B)
        DTSS4= -WSTLORG(B)*MAX(SEDORG(B),0.)
        DTSCLY(B)=DTESCLY(B)+DTSS1/BL(B,3)  ! G/M^3/S
        DTSSLT(B)=DTESSLT(B)+DTSS2/BL(B,3)
        DTSSND(B)=DTESSND(B)+DTSS3/BL(B,3)
        DTSORG(B)=DTESORG(B)+DTSS4/BL(B,3)
      END DO

! NOW SUB-SURFACE BOXES (BU(B) > 0)
      DO B=NSB+1,NB
        DTSS1=WSTLCLY(BU(B))*SEDCLY(BU(B))   &
              -WSTLCLY(B)*SEDCLY(B) ! G/M^2/S  &
        DTSS2=MIN(WSTLSLT(BU(B)),BL(BU(B),3)/DLT)*SEDSLT(BU(B))  &
              -MIN(WSTLSLT(B),BL(B,3)/DLT)*SEDSLT(B)
        DTSS3=MIN(WSTLSND(BU(B)),BL(BU(B),3)/DLT)*SEDSND(BU(B))  &
              -MIN(WSTLSND(B),BL(B,3)/DLT)*SEDSND(B)
        DTSS4=WSTLORG(BU(B))*MAX(SEDORG(BU(B)),0.)  &
              -WSTLORG(B)*MAX(SEDORG(B),0.)
        DTSCLY(B)=DTESCLY(B)+DTSS1/BL(B,3)  ! G/M^3/S
        DTSSLT(B)=DTESSLT(B)+DTSS2/BL(B,3)
        DTSSND(B)=DTESSND(B)+DTSS3/BL(B,3)
        DTSORG(B)=DTESORG(B)+DTSS4/BL(B,3)
      END DO

! SETTLING FLUX FOR MASS BALANCE   g/s

      IF (S_TRANS_FLUX) THEN
        DO F=NHQF+1,NQF
          B = JB(F)
          FLXSCLAY(F) = -WSTLCLY(B)*SEDCLY(B)*V2(B)/BL(B,3)    ! G/S
          FLXSSILT(F) = -MIN(WSTLSLT(B),BL(B,3)/DLT)*SEDSLT(B)  &
                        *V2(B)/BL(B,3)
          FLXSSAND(F) = -MIN(WSTLSND(B),BL(B,3)/DLT)*SEDSND(B)  &
                        *V2(B)/BL(B,3)
          FLXSORGM(F) = -WSTLORG(B)*MAX(SEDORG(B),0.)*V2(B)/BL(B,3)
        END DO
      END IF
        
      RETURN
      END
